(() => {
  const header = document.querySelector(".header");
  const burger = document.getElementById("burger");
  const nav = document.getElementById("nav");

  // ===== Helpers =====
  const getHeaderOffset = () => (header ? header.offsetHeight : 0);

  const scrollToHash = (hash, smooth = true) => {
    const el = document.querySelector(hash);
    if (!el) return;

    const top =
      el.getBoundingClientRect().top + window.pageYOffset - getHeaderOffset() + 1;

    const reduceMotion =
      window.matchMedia?.("(prefers-reduced-motion: reduce)")?.matches;

    window.scrollTo({
      top,
      behavior: !smooth || reduceMotion ? "auto" : "smooth",
    });
  };

  const setActiveLink = (hash) => {
    document.querySelectorAll(".nav__link").forEach((l) => {
      l.classList.toggle("is-active", l.getAttribute("href") === hash);
    });
  };

  // ===== Mobile menu =====
  if (burger && nav) {
    burger.addEventListener("click", () => nav.classList.toggle("is-open"));

    document.addEventListener("click", (e) => {
      const isClickNav = nav.contains(e.target);
      const isClickBurger = burger.contains(e.target);
      if (!isClickNav && !isClickBurger) nav.classList.remove("is-open");
    });
  }

  // ===== Footer year =====
  const year = document.getElementById("year");
  if (year) year.textContent = new Date().getFullYear();

  // ===== Form submit (demo) =====
  // ===== Form submit (mailto) =====
  const form = document.getElementById("dealerForm");
  const formMsg = document.getElementById("formMsg");

  if (form && formMsg) {
    form.addEventListener("submit", (e) => {
      e.preventDefault();

      const data = new FormData(form);
      const name = String(data.get("name") || "").trim();
      const email = String(data.get("email") || "").trim();
      const phone = String(data.get("phone") || "").trim();
      const note = String(data.get("note") || "").trim();

      if (!name || !email || !phone) {
        formMsg.textContent = "Vui lòng nhập đủ Họ tên / Email / Số điện thoại.";
        formMsg.className = "form__msg danger";
        return;
      }

      // ✅ đổi email nhận tại đây
      const to = "info.helionlink.com@gmail.com";

      const subject = encodeURIComponent("Đăng ký đại lý / Liên hệ");
      const body = encodeURIComponent(
        `Họ và tên: ${name}\nEmail: ${email}\nSĐT: ${phone}\nGhi chú: ${note}`
      );

      window.location.href = `mailto:${to}?subject=${subject}&body=${body}`;

      formMsg.textContent = "Đã mở ứng dụng email. Vui lòng bấm Gửi để hoàn tất.";
      formMsg.className = "form__msg ok";
    });
  }


  // ===== Smooth scroll + active menu =====
  document.querySelectorAll('a[href^="#"]').forEach((a) => {
    a.addEventListener("click", (e) => {
      const hash = a.getAttribute("href");
      if (!hash || hash === "#") return;

      const el = document.querySelector(hash);
      if (!el) return;

      e.preventDefault();
      scrollToHash(hash, true);
      setActiveLink(hash);
      nav?.classList.remove("is-open");
    });
  });

  // ===== Active menu khi scroll (IntersectionObserver) =====
  const sections = Array.from(document.querySelectorAll("section[id], footer[id]"));
  let observer = null;

  const setActiveByPosition = () => {
    const offset = getHeaderOffset() + 10;
    const current = sections.find((s) => {
      const rect = s.getBoundingClientRect();
      return rect.top <= offset && rect.bottom > offset;
    });
    if (current?.id) setActiveLink(`#${current.id}`);
  };

  const createObserver = () => {
    if (!sections.length) return;
    observer?.disconnect();

    observer = new IntersectionObserver(
      (entries) => {
        const visible = entries
          .filter((en) => en.isIntersecting)
          .sort((a, b) => b.intersectionRatio - a.intersectionRatio)[0];

        if (visible?.target?.id) setActiveLink(`#${visible.target.id}`);
      },
      {
        root: null,
        rootMargin: `-${getHeaderOffset()}px 0px -55% 0px`,
        threshold: [0.15, 0.25, 0.4, 0.6],
      }
    );

    sections.forEach((s) => observer.observe(s));
  };

  // =========================================================
  // ===== Partners Carousel loop (prev/next) - NO GAP ========
  // (clone theo số card đang nhìn thấy để không bao giờ hở)
  // =========================================================
  function initPartnersCarousel() {
    const track = document.getElementById("pcTrack");
    const viewport = document.getElementById("pcViewport");
    const btnPrev = document.getElementById("pcPrev");
    const btnNext = document.getElementById("pcNext");
    if (!track || !viewport || !btnPrev || !btnNext) return;

    let realItems = [];
    let clonesPerSide = 1;
    let index = 0;
    let isAnimating = false;
    let fallbackTimer = null;

    const mod = (n, m) => ((n % m) + m) % m;

    const cleanClones = () => {
      track.querySelectorAll('[data-clone="1"]').forEach((n) => n.remove());
    };

    const getGap = () => {
      const style = window.getComputedStyle(track);
      const g = style.gap || style.columnGap || "0px";
      return parseFloat(g) || 0;
    };

    const getStep = () => {
      const first = realItems[0];
      if (!first) return 0;
      return first.getBoundingClientRect().width + getGap();
    };

    const getVisibleCount = () => {
      const step = getStep();
      if (!step) return 1;

      const cs = window.getComputedStyle(viewport);
      const padL = parseFloat(cs.paddingLeft) || 0;
      const padR = parseFloat(cs.paddingRight) || 0;
      const innerW = viewport.clientWidth - padL - padR;

      // ceil + 1 để không hở khi đang chạy
      return Math.max(1, Math.ceil(innerW / step) + 1);
    };

    const apply = (animate = true) => {
      const step = getStep();
      if (!step) return;

      track.style.transition = animate ? "transform 380ms ease" : "none";
      track.style.transform = `translateX(${-index * step}px)`;
    };

    const unlockFallback = () => {
      clearTimeout(fallbackTimer);
      fallbackTimer = setTimeout(() => (isAnimating = false), 450);
    };

    const snapIfNeeded = () => {
      const realCount = realItems.length;
      const minReal = clonesPerSide;
      const maxReal = clonesPerSide + realCount - 1;

      if (index < minReal) {
        index += realCount;
        apply(false);
      } else if (index > maxReal) {
        index -= realCount;
        apply(false);
      }
    };

    const go = (dir) => {
      if (isAnimating) return;
      isAnimating = true;

      index += dir;
      apply(true);
      unlockFallback();
    };

    const build = () => {
      realItems = Array.from(track.children).filter((el) => !el.dataset.clone);
      const realCount = realItems.length;
      if (realCount < 2) return;

      cleanClones();

      // số clone không được vượt quá số item thật
      clonesPerSide = Math.min(getVisibleCount(), realCount);

      // clone trái: lấy N item cuối
      for (let i = 0; i < clonesPerSide; i++) {
        const src = realItems[mod(realCount - clonesPerSide + i, realCount)];
        const c = src.cloneNode(true);
        c.dataset.clone = "1";
        track.insertBefore(c, track.firstChild);
      }

      // clone phải: lấy N item đầu
      for (let i = 0; i < clonesPerSide; i++) {
        const src = realItems[mod(i, realCount)];
        const c = src.cloneNode(true);
        c.dataset.clone = "1";
        track.appendChild(c);
      }

      index = clonesPerSide;
      requestAnimationFrame(() => apply(false));
    };

    // bind 1 lần (dùng onclick để không addEventListener chồng)
    btnNext.onclick = () => go(1);
    btnPrev.onclick = () => go(-1);

    track.addEventListener("transitionend", () => {
      clearTimeout(fallbackTimer);
      snapIfNeeded();
      requestAnimationFrame(() => (isAnimating = false));
    });

    // đợi ảnh load xong rồi build để đo step đúng
    const imgs = Array.from(track.querySelectorAll("img"));
    const pending = imgs.filter((img) => !img.complete).length;

    if (!pending) {
      build();
    } else {
      let left = pending;
      imgs.forEach((img) => {
        if (img.complete) return;
        img.addEventListener(
          "load",
          () => {
            left--;
            if (left === 0) build();
          },
          { once: true }
        );
      });
    }

    // resize -> rebuild (debounce)
    let t;
    window.addEventListener("resize", () => {
      clearTimeout(t);
      t = setTimeout(build, 150);
    });
  }
  // ===== Back to top =====
  const backToTop = document.getElementById("backToTop");

  if (backToTop) {
    const toggle = () => {
      backToTop.classList.toggle("show", window.scrollY > 400);
    };

    toggle(); // chạy ngay lần đầu
    window.addEventListener("scroll", toggle);

    backToTop.addEventListener("click", () => {
      const reduceMotion =
        window.matchMedia?.("(prefers-reduced-motion: reduce)")?.matches;

      window.scrollTo({ top: 0, behavior: reduceMotion ? "auto" : "smooth" });
    });
  } else {
    console.warn("Không tìm thấy #backToTop trong HTML");
  }
  // ===== Popup hướng dẫn cài đặt + tải zip =====
  const openInstall = document.getElementById("openInstall");
  const installModal = document.getElementById("installModal");

  function openModal(modal) {
    if (!modal) return;
    modal.classList.add("is-open");
    modal.setAttribute("aria-hidden", "false");
    document.body.style.overflow = "hidden";
  }

  function closeModal(modal) {
    if (!modal) return;
    modal.classList.remove("is-open");
    modal.setAttribute("aria-hidden", "true");
    document.body.style.overflow = "";
  }

  if (openInstall && installModal) {
    openInstall.addEventListener("click", () => openModal(installModal));

    // close when click overlay or button [data-close]
    installModal.addEventListener("click", (e) => {
      const target = e.target;
      if (target?.dataset?.close === "1") closeModal(installModal);
    });

    // ESC to close
    document.addEventListener("keydown", (e) => {
      if (e.key === "Escape" && installModal.classList.contains("is-open")) {
        closeModal(installModal);
      }
    });
  }


  // ===== Load / hash =====
  window.addEventListener("load", () => {
    createObserver();

    if (location.hash && document.querySelector(location.hash)) {
      scrollToHash(location.hash, false);
      setActiveLink(location.hash);
    } else {
      setActiveLink("#ve-chung-toi");
      setActiveByPosition();
    }

    // init carousel sau khi layout/ảnh có thể đã render
    initPartnersCarousel();
  });

  window.addEventListener("hashchange", () => {
    if (location.hash && document.querySelector(location.hash)) {
      scrollToHash(location.hash, true);
      setActiveLink(location.hash);
    }
  });

})();
